<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor color control.
 *
 * A base control for creating color control. Displays a color picker with an
 * alpha slider. Includes a customizable color palette that can be preset by the
 * user. Accepts a `scheme` argument that allows you to set a value from the
 * active color scheme as the default value returned by the control.
 *
 * Creating new control in the editor (inside `Widget_Base::_register_controls()`
 * method):
 *
 *    $this->add_control(
 *    	'title_color',
 *    	[
 *    		'label' => __( 'Title Color', 'plugin-domain' ),
 *    		'type' => Controls_Manager::COLOR,
 *    		'scheme' => [
 *    			'type' => Scheme_Color::get_type(),
 *    			'value' => Scheme_Color::COLOR_1,
 *    		],
 *    		'selectors' => [
 *    			'{{WRAPPER}} .title' => 'color: {{VALUE}}',
 *    		],
 *    	]
 *    );
 *
 * PHP usage (inside `Widget_Base::render()` method):
 *
 *    echo '<h2 class="title" style="color:' . $this->get_settings( 'title_color' ) . '"> ... </h2>';
 *
 * JS usage (inside `Widget_Base::_content_template()` method):
 *
 *    <h2 class="title" style="color:{{ settings.title_color }}"> ... </h2>
 *
 * @since 1.0.0
 *
 * @param string $label       Optional. The label that appears above of the
 *                            field. Default is empty.
 * @param string $description Optional. The description that appears below the
 *                            field. Default is empty.
 * @param string $default     Optional. Default color in rgb, rgba, or hex format.
 *                            Default is empty.
 * @param array  $scheme      Optional. The value from the active color scheme
 *                            as the default value returned by the control.
 * @param bool   $alpha       Optional. Whether to allow alpha channel. Default
 *                            is true.
 * @param string $separator   Optional. Set the position of the control separator.
 *                            Available values are 'default', 'before', 'after'
 *                            and 'none'. 'default' will position the separator
 *                            depending on the control type. 'before' / 'after'
 *                            will position the separator before/after the
 *                            control. 'none' will hide the separator. Default
 *                            is 'default'.
 * @param bool   $show_label  Optional. Whether to display the label. Default is
 *                            true.
 * @param bool   $label_block Optional. Whether to display the label in a
 *                            separate line. Default is false.
 */
class Control_Color extends Base_Data_Control {

	/**
	 * Retrieve color control type.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Control type.
	 */
	public function get_type() {
		return 'color';
	}

	/**
	 * Enqueue color control scripts and styles.
	 *
	 * Used to register and enqueue custom scripts and styles used by the color
	 * control.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function enqueue() {
		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		wp_register_script(
			'wp-color-picker-alpha',
			ELEMENTOR_ASSETS_URL . 'lib/wp-color-picker/wp-color-picker-alpha' . $suffix . '.js',
			[
				'wp-color-picker',
			],
			'2.0.1',
			true
		);

		global $wp_version;
		if ( version_compare( $wp_version, '5.5', '>=' ) ) {
			$color_picker_strings = array(
				'clear'            => __( 'Clear', 'elementor' ),
				'clearAriaLabel'   => __( 'Clear color', 'elementor' ),
				'defaultString'    => __( 'Default', 'elementor' ),
				'defaultAriaLabel' => __( 'Select default color', 'elementor' ),
				'pick'             => __( 'Select Color', 'elementor' ),
				'defaultLabel'     => __( 'Color value', 'elementor' ),
			);
			wp_localize_script( 'wp-color-picker-alpha', 'wpColorPickerL10n', $color_picker_strings );
		}

		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'wp-color-picker-alpha' );
	}

	/**
	 * Render color control output in the editor.
	 *
	 * Used to generate the control HTML in the editor using Underscore JS
	 * template. The variables for the class are available using `data` JS
	 * object.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function content_template() {
		?>
		<# var defaultValue = '', dataAlpha = '';
			if ( data.default ) {
				defaultValue = ' data-default-color=' + data.default; // Quotes added automatically.
			}

			if ( data.alpha ) {
				dataAlpha = ' data-alpha=true';
			} #>
		<div class="elementor-control-field">
			<label class="elementor-control-title">
				<# if ( data.label ) { #>
					{{{ data.label }}}
				<# } #>
				<# if ( data.description ) { #>
					<span class="elementor-control-field-description">{{{ data.description }}}</span>
				<# } #>
			</label>
			<div class="elementor-control-input-wrapper">
				<input data-setting="{{ name }}" type="text" placeholder="<?php echo esc_attr( 'Hex/rgba', 'elementor' ); ?>" {{ defaultValue }}{{ dataAlpha }} />
			</div>
		</div>
		<?php
	}

	/**
	 * Retrieve color control default settings.
	 *
	 * Get the default settings of the color control. Used to return the default
	 * settings while initializing the color control.
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @return array Control default settings.
	 */
	protected function get_default_settings() {
		return [
			'alpha' => true,
		];
	}
}
