<?php
/**
 * Handle order via single product
 */
namespace Webforia_Whatsapp;

class Order
{
    public function __construct()
    {
        add_action('add_meta_boxes', [$this, 'register_metabox']);
        add_filter('manage_edit-shop_order_columns', [$this, 'admin_list_column']);
        add_action('manage_shop_order_posts_custom_column', [$this, 'admin_list_column_content']);
        add_action('wp_enqueue_scripts', [$this, 'order_scripts']);
        add_action('wc_ajax_whatsapp_create_order', [$this, 'create_order_result']);
        add_shortcode('wwc_single_add_to_cart', [$this, 'wwc_single_add_to_cart']);

    }
    /**
     * Add new box order detail
     *
     * @return void
     */
    public function register_metabox()
    {
        $order = wc_get_order(get_the_ID());

        if (!empty($_GET['action']) && $_GET['action'] == 'edit' && get_post_type() == 'shop_order') {

            if ($order->get_status() == 'on-hold' || $order->get_status() == 'pending') {
                add_meta_box("Follow Up", "Follow Up", [$this, 'metabox'], "shop_order", "side", "high", null);
            }
        }
    }

    /**
     * Get message from setting
     *
     * @param [type] $order_id
     * @return message
     */
    public function get_order_status_message($order_id)
    {
        $order = wc_get_order($order_id);
        $phone = wwc_phone_format($order->get_billing_phone());

        // check order status
        $message = '';

        if ($order->get_status() == 'on-hold' || $order->get_status() == 'pending') {
            $message = wwc_option('whatsapp_follow_up_onhold');
        }

        return $this->create_whatsapp_link($order_id, $message, $phone);

    }

    /**
     * Show button on single order page
     *
     * @return void
     */
    public function metabox()
    {
        $link = $this->get_order_status_message(get_the_ID());

        if (!empty($link)) {
            echo wp_sprintf('<span class="bulma"><a href="%s" target="_blank" class="button is-success is-fullwidth">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-whatsapp" viewBox="0 0 16 16" style="margin-right: 5px;">
                    <path d="M13.601 2.326A7.854 7.854 0 0 0 7.994 0C3.627 0 .068 3.558.064 7.926c0 1.399.366 2.76 1.057 3.965L0 16l4.204-1.102a7.933 7.933 0 0 0 3.79.965h.004c4.368 0 7.926-3.558 7.93-7.93A7.898 7.898 0 0 0 13.6 2.326zM7.994 14.521a6.573 6.573 0 0 1-3.356-.92l-.24-.144-2.494.654.666-2.433-.156-.251a6.56 6.56 0 0 1-1.007-3.505c0-3.626 2.957-6.584 6.591-6.584a6.56 6.56 0 0 1 4.66 1.931 6.557 6.557 0 0 1 1.928 4.66c-.004 3.639-2.961 6.592-6.592 6.592zm3.615-4.934c-.197-.099-1.17-.578-1.353-.646-.182-.065-.315-.099-.445.099-.133.197-.513.646-.627.775-.114.133-.232.148-.43.05-.197-.1-.836-.308-1.592-.985-.59-.525-.985-1.175-1.103-1.372-.114-.198-.011-.304.088-.403.087-.088.197-.232.296-.346.1-.114.133-.198.198-.33.065-.134.034-.248-.015-.347-.05-.099-.445-1.076-.612-1.47-.16-.389-.323-.335-.445-.34-.114-.007-.247-.007-.38-.007a.729.729 0 0 0-.529.247c-.182.198-.691.677-.691 1.654 0 .977.71 1.916.81 2.049.098.133 1.394 2.132 3.383 2.992.47.205.84.326 1.129.418.475.152.904.129 1.246.08.38-.058 1.171-.48 1.338-.943.164-.464.164-.86.114-.943-.049-.084-.182-.133-.38-.232z"></path>
                </svg> %s</a></span>', $link, __('Follow Up', WEBFORIA_WHATSAPP_CHAT_DOMAIN));
        } else {
            echo __('Nomor WhatsApp tidak tersedia', WEBFORIA_WHATSAPP_CHAT_DOMAIN);
        }
    }

    /**
     * Create new column on order list admin
     *
     * @param [type] $columns
     * @return void
     */
    public function admin_list_column($columns)
    {
        $columns['follow_up'] = __('Follow Up', WEBFORIA_WHATSAPP_CHAT_DOMAIN);
        return $columns;
    }

    /**
     * Content list
     *
     * @param [type] $column
     * @return void
     */
    public function admin_list_column_content($column)
    {
     
        $link  = $this->get_order_status_message(get_the_ID());
        $order = wc_get_order(get_the_ID());

        if ('follow_up' === $column && !empty($link)) {
            if ($order->get_status() == 'on-hold' || $order->get_status() == 'pending') {
                echo wp_sprintf('<a href="%s" target="_blank" class="button wc-action-button">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-whatsapp" viewBox="0 0 16 16" style="position: relative; bottom: -2px; margin-right: 5px;">
                    <path d="M13.601 2.326A7.854 7.854 0 0 0 7.994 0C3.627 0 .068 3.558.064 7.926c0 1.399.366 2.76 1.057 3.965L0 16l4.204-1.102a7.933 7.933 0 0 0 3.79.965h.004c4.368 0 7.926-3.558 7.93-7.93A7.898 7.898 0 0 0 13.6 2.326zM7.994 14.521a6.573 6.573 0 0 1-3.356-.92l-.24-.144-2.494.654.666-2.433-.156-.251a6.56 6.56 0 0 1-1.007-3.505c0-3.626 2.957-6.584 6.591-6.584a6.56 6.56 0 0 1 4.66 1.931 6.557 6.557 0 0 1 1.928 4.66c-.004 3.639-2.961 6.592-6.592 6.592zm3.615-4.934c-.197-.099-1.17-.578-1.353-.646-.182-.065-.315-.099-.445.099-.133.197-.513.646-.627.775-.114.133-.232.148-.43.05-.197-.1-.836-.308-1.592-.985-.59-.525-.985-1.175-1.103-1.372-.114-.198-.011-.304.088-.403.087-.088.197-.232.296-.346.1-.114.133-.198.198-.33.065-.134.034-.248-.015-.347-.05-.099-.445-1.076-.612-1.47-.16-.389-.323-.335-.445-.34-.114-.007-.247-.007-.38-.007a.729.729 0 0 0-.529.247c-.182.198-.691.677-.691 1.654 0 .977.71 1.916.81 2.049.098.133 1.394 2.132 3.383 2.992.47.205.84.326 1.129.418.475.152.904.129 1.246.08.38-.058 1.171-.48 1.338-.943.164-.464.164-.86.114-.943-.049-.084-.182-.133-.38-.232z"></path>
                </svg>
                %s</a>', $link, __('Follow Up', WEBFORIA_WHATSAPP_CHAT_DOMAIN));
            }
        }
    }

    /**
     * Get detail address form order
     *
     * @param [type] $order_id
     * @return array
     */
    public function get_formatted_address($order_id)
    {
        $order = wc_get_order($order_id);

        // Order
        $order_number     = $order->get_order_number();
        $order_date       = wc_format_datetime($order->get_date_created());
        $order_first_name = $order->get_billing_first_name();
        $order_last_name  = $order->get_billing_last_name();

        //Biling
        $billing = wp_sprintf(__('Nama: %s', WEBFORIA_WHATSAPP_CHAT_DOMAIN), $order->get_formatted_billing_full_name()) . '%0A';

        if ($order->get_billing_email()) {
            $billing .= wp_sprintf(__('Email: %s', WEBFORIA_WHATSAPP_CHAT_DOMAIN), $order->get_billing_email()) . '%0A';

        }

        if ($order->get_billing_phone()) {
            $billing .= wp_sprintf(__('Phone: %s', WEBFORIA_WHATSAPP_CHAT_DOMAIN), $order->get_billing_phone()) . '%0A';
        }

        // shipping
        $shipping = $order->get_formatted_shipping_address() . '%0A';

        // product
        $product = '';
        $order_detail = '';
        $index        = 1;
        foreach ($order->get_items() as $item_id => $item) {

            // get order item data (in an unprotected array)
            $item_data = $item->get_data();

            // get order item meta data (in an unprotected array)
            $item_meta_data = $item->get_meta_data();

            // get only All item meta data even hidden (in an unprotected array)
            $formatted_meta_data = $item->get_formatted_meta_data('_', true);
            $array               = json_decode(json_encode($formatted_meta_data), true);
            $arrayx              = array_values($array);
            $arrayxxx            = array_merge($array);
            $result              = array();
           

            $product_name = $item->get_name();

            $order_detail .= wp_sprintf('*%s. %s*', $index, $product_name) . '%0A';

            // product meta
            if(wwc_option('product_item_meta_data', true)){
                foreach ((array) $arrayxxx as $value) {
                    $product_meta = "";
                    $result[]     = array($value["display_key"], wp_strip_all_tags($value["display_value"]));
                    foreach ($result as $key) {
                        $result = array();
                        $product_meta .= " - " . $key[0] . " : " . $key[1] . "\r\n";
                        $order_detail .=  urlencode("" . $product_meta . "");
                    }

                    $order_detail .= '%0A';
                }
            }

            $order_detail .= wp_sprintf(__('Jumlah: %s', WEBFORIA_WHATSAPP_CHAT_DOMAIN), $item->get_quantity()) . '%0A';
            $order_detail .= wp_sprintf(__('Harga(@): %s', WEBFORIA_WHATSAPP_CHAT_DOMAIN), wwc_price($item->get_subtotal() / $item->get_quantity())) . '%0A';
            $order_detail .= wp_sprintf(__('Sub Total: %s', WEBFORIA_WHATSAPP_CHAT_DOMAIN), wwc_price($item->get_subtotal())) . '%0A%0A';
            $index++;
        }

        $order_detail .= wp_sprintf(__('Sub Total: *%s*', WEBFORIA_WHATSAPP_CHAT_DOMAIN), wwc_price($order->get_subtotal())) . '%0A';

        if ($order->get_total_discount()) {
            $order_detail .= wp_sprintf(__('Diskon: *- %s*', WEBFORIA_WHATSAPP_CHAT_DOMAIN), wwc_price($order->get_total_discount())) . '%0A';
        }

        if ($order->get_shipping_total()) {
            $order_detail .= wp_sprintf(__('Ongkos kirim: *%s* - %s', WEBFORIA_WHATSAPP_CHAT_DOMAIN), wwc_price($order->get_shipping_total()), $order->get_shipping_method()) . '%0A';
        }

        if ($order->get_payment_method_title()) {
            $order_detail .= wp_sprintf(__('Metode Pembayaran: *%s*', WEBFORIA_WHATSAPP_CHAT_DOMAIN), $order->get_payment_method_title()) . '%0A';
        }

        $order_detail .= wp_sprintf(__('Total: *%s*', WEBFORIA_WHATSAPP_CHAT_DOMAIN), wwc_price($order->get_total())) . '%0A';

        // bank account
        $bank = '';

        if (!empty(get_option('woocommerce_bacs_accounts')) && $order->get_payment_method() == 'bacs') {
            $bank = '*' . __('Nomor rekening', WEBFORIA_WHATSAPP_CHAT_DOMAIN) . '*' . '%0A';

            foreach (get_option('woocommerce_bacs_accounts') as $key => $bacs) {
                $bank .= "*{$bacs['bank_name']}* - {$bacs['account_name']} - {$bacs['account_number']}" . '%0A';
            }
        }

        //notes
        if ($order->get_customer_note()) {
            $order_note = wp_kses_post(nl2br(wptexturize($order->get_customer_note())));
        } else {
            $order_note = '-';
        }

        return array(
            'order_number' => $order_number,
            'order_date'   => $order_date,
            'first_name'    => $order_first_name,
            'last_name'    => $order_last_name,
            'billing'      => $billing,
            'shipping'     => $shipping,
            'order_detail' => $order_detail,
            'product'      => $product,
            'bank'         => $bank,
            'note'         => $order_note,
            'link'         => get_the_permalink(),
        );
    }

    /**
     * Create whatsapp link form order detail
     * Show order detail with whatsapp format
     *
     * @param [type] $order_id
     * @param [type] $message
     * @param string $number
     * @return string
     */
    public function create_whatsapp_link($order_id, $message = '', $number = '')
    {

        $order = $this->get_formatted_address($order_id);

        $phone_admin = !empty($number) ? $number : wwc_rotator_number();

        $text_map = array('{{order_number}}', '{{order_date}}', '{{first_name}}', '{{last_name}}', '{{billing}}', '{{shipping}}', '{{product}}', '{{order_detail}}', '{{bank}}', '{{note}}', '{{link}}');

        $text_value = array($order['order_number'], $order['order_date'], $order['first_name'], $order['last_name'], $order['billing'], $order['shipping'], $order['product'], $order['order_detail'], $order['bank'], $order['note'], $order['link']);

        $message = wwc_whatsapp_text_format(str_replace($text_map, $text_value, $message));

        return "https://api.whatsapp.com/send?phone={$phone_admin}&text={$message}";

    }

    /**
     * Ajax Scripts
     *
     * @return void
     */
    public function order_scripts()
    {

        wp_enqueue_script('webforia-whatsapp-order', WEBFORIA_WHATSAPP_CHAT_ASSETS . 'assets/js/create-order.min.js', array('jquery'), '1.3.0', true);
        wp_localize_script('webforia-whatsapp-order', 'order_var',
            array(
                'check_nonce' => wp_create_nonce('webforia-whatsapp-order-nonce'),
            )
        );

    }

    /**
     * Save new order on database woocommerce
     */
    public function create_order_result()
    {

        parse_str($_POST['field'], $args);

        $post = wp_parse_args($args, array(
            'first_name'     => '',
            'last_name'      => '',
            'email'          => '',
            'phone'          => '',
            'address'        => '',
            'order_checked'  => '',
            'note'           => '',
            'payment_method' => 'bacs',
        ));

        $fullname      = wwc_split_fullname($post['name']);
        $options       = wwc_option('whatsapp_product_field', 'option');
        $product_id    = intval($_POST['product_id']);
        $quantity      = intval($_POST['quantity']);
        $order_id      = '';
        $whatsapp_link = '';
        $country_id    = 'ID';
        $shipping_cost = apply_filters('wwc_shipping_cost', 0);

        // Validation
        $errors = false;

        if (empty($post['name']) && in_array('fullname', $options)) {
            wc_add_notice(__('Name is a required field', WEBFORIA_WHATSAPP_CHAT_DOMAIN), 'error');
            $errors = true;
        }

        if (empty($post['phone']) && in_array('phone', $options)) {
            wc_add_notice(__('Phone is a required field', WEBFORIA_WHATSAPP_CHAT_DOMAIN), 'error');
            $errors = true;
        }

        if (!intval($post['phone']) && !empty($post['phone']) && in_array('phone', $options)) {
            wc_add_notice(__('Invalid phone number', WEBFORIA_WHATSAPP_CHAT_DOMAIN), 'error');
            $errors = true;
        }

        if (empty($post['email']) && in_array('email', $options)) {
            wc_add_notice(__('Email is a required field', WEBFORIA_WHATSAPP_CHAT_DOMAIN), 'error');
            $errors = true;
        }

        if (!is_email($post['email']) && !empty($post['email']) && in_array('email', $options)) {
            wc_add_notice(__('Invalid email address', WEBFORIA_WHATSAPP_CHAT_DOMAIN), 'error');
            $errors = true;
        }

        if (empty($post['address']) && in_array('address', $options)) {
            wc_add_notice(__('Address is a required field', WEBFORIA_WHATSAPP_CHAT_DOMAIN), 'error');
            $errors = true;
        }

        if (empty($errors)) {

            $addresses = array(
                'first_name' => $fullname['first_name'],
                'last_name'  => $fullname['last_name'],
                'email'      => $post['email'],
                'phone'      => $post['phone'],
                'address_1'  => $post['address'],
                'country'    => $country_id,
            );

            // Create order
            $order = wc_create_order();
            $order->set_customer_id(get_current_user_id());
            $order->add_product(wc_get_product($product_id), $quantity);

            // save billing and shipping
            foreach ($addresses as $key => $address) {
                update_post_meta($order->get_id(), "_billing_{$key}", $address);
                update_post_meta($order->get_id(), "_shipping_{$key}", $address);
            }
            update_post_meta($order->get_id(), 'shipping_method', true);

            $order->set_payment_method($post['payment_method']);
            $order->set_customer_ip_address(\WC_Geolocation::get_ip_address());
            $order->set_customer_user_agent(wc_get_user_agent());
            // $order->add_order_note(_('Order from product WhatsApp button', WEBFORIA_WHATSAPP_CHAT_DOMAIN));

            if ($post['order_checked'] && !empty($post['note'])) {
                $order->set_customer_note($post['note']);
            }

            // Shipping Fees
            if (!empty($shipping_cost) && $shipping_cost > 0) {
                $item = new \WC_Order_Item_Shipping();
                $item->set_method_title("Flat rate");
                $item->set_method_id("flat_rate:14"); // set an existing Shipping method rate ID
                $item->set_total($shipping_cost); // (optional)
                $item->calculate_taxes(array('country' => $country_id));
                $order->add_item($item);
            }

            // Calculate
            $order->calculate_totals();
            $order->update_status('on-hold');

            $order_id = $order->get_id();

            wc_add_notice(__('Thank you. Your order has been received.', WEBFORIA_WHATSAPP_CHAT_DOMAIN));
        }

        // Generate link
        if (!empty($order_id)) {
            $whatsapp_link = $this->create_whatsapp_link($order_id, wwc_option('whatsapp_product_message'));
        }

        // Error message
        ob_start();
        wc_print_notices();
        $message = ob_get_clean();

        // Send
        wp_send_json([
            'link'    => $whatsapp_link,
            'error'   => $errors,
            'message' => $message,
        ]);

    }

    /* Pixel Caffein Support
     *
     * fired after add to cart ajax
     * @hook action woocommerce_ajax_added_to_cart
     * @see https://wordpress.org/support/topic/integrate-with-popup-maker-for-automatic-event-tracking/
     */
    public function pixel_caffein($id)
    {
        $product = wc_get_product($id);

        if (function_exists('aepc_pixel_event_code')) {
            aepc_pixel_event_code([
                'event_name'        => 'AddToCart', // required
                'value'             => $product->get_price(),
                'content_name'      => $product->get_name(),
                'content_category ' => $product->get_categories(),
                'content_ids'       => $product->get_id(),
                'content_type'      => 'product',
                'value'             => $product->get_price(),
                'currency'          => get_woocommerce_currency_symbol(),
            ]);

        }

    }

    /**
     * Checkout modal on page anywhere
     * @shortcode
     */
    public function wwc_single_add_to_cart($atts, $content, $shortcode_tag)
    {
        $default_atts = array(
            "product_id"  => 1,
            "text"        => wwc_option('whatsapp_product_text'),
            'pixel'       => wwc_option('whatsapp_product_pixel_event'),
            'quantity'    => 1,
            'field'       => wwc_option('whatsapp_product_field', 'option'),
            'modal_title' => wwc_option('whatsapp_product_modal_title'),
            'modal_desc'  => wwc_option('whatsapp_product_desc'),
        );

        $params = shortcode_atts($default_atts, $atts);

        ob_start();

        wwc_get_template_part('one-page-order', $params);

        $modal_form = ob_get_clean();

        return $modal_form;
    }

}
